import 'package:rad/rad.dart';
import 'package:rad/widgets_async.dart';
import 'package:rad/widgets_html.dart';

import 'package:photogram_admin_cp/import/core.dart';
import 'package:photogram_admin_cp/import/data.dart';

import 'package:photogram_admin_cp/ui/components/util_widgets.dart';

class GraphItem {
  String title;
  String color;
  AdminTableStatsDTO tableStatsDTO;

  GraphItem({
    required this.title,
    required this.color,
    required this.tableStatsDTO,
  });
}

class DashboardPage extends StatefulWidget {
  const DashboardPage({Key? key}) : super(key: key);

  @override
  _DashboardPageState createState() => _DashboardPageState();
}

class _DashboardPageState extends State<DashboardPage> {
  late final _calculation = Future<ResponseModel>(() async {
    return await AppProvider.of(context).apiRepo.preparedRequest(
      requestType: REQ_TYPE_ADMIN_TABLE_STATS,
      requestData: {},
    );
  });

  @override
  build(context) {
    return FutureBuilder<ResponseModel>(
      future: _calculation,
      builder: (context, snapshot) {
        if (snapshot.hasData) {
          var responseModel = snapshot.data!;

          AdminTableStatsDTO userStats;
          AdminTableStatsDTO postStats;
          AdminTableStatsDTO postLikeStats;
          AdminTableStatsDTO postCommentStats;
          AdminTableStatsDTO hashtagStats;

          var emptyDTO = AdminTableStatsDTO.fromJson({});

          userStats = postStats = postLikeStats = postCommentStats = hashtagStats = emptyDTO;

          for (Map<String, dynamic> dtoData in responseModel.content[AdminTableStatsDTO.dtoName]) {
            var statsDTO = AdminTableStatsDTO.fromJson(dtoData);

            switch (statsDTO.tableName) {
              case UserTable.tableName:
                userStats = statsDTO;
                break;

              case PostTable.tableName:
                postStats = statsDTO;
                break;

              case PostLikeTable.tableName:
                postLikeStats = statsDTO;
                break;

              case PostCommentTable.tableName:
                postCommentStats = statsDTO;
                break;

              case HashtagTable.tableName:
                hashtagStats = statsDTO;
                break;
            }
          }

          return Division(
            className: 'content-fluid animate__animated animate__fadeIn',
            child: Division(
              className: 'row',
              children: [
                // main side containing charts

                Division(
                  className: 'col-md-12 col-lg-8 col-xl-6',
                  children: [
                    if (postLikeStats.isDTO || postCommentStats.isDTO || hashtagStats.isDTO)
                      _buildGraph(
                        title: 'Interactions',
                        type: 'bar',
                        graphItems: [
                          if (postLikeStats.isDTO)
                            GraphItem(
                              title: 'Post likes',
                              color: 'rgb(255, 205, 86)',
                              tableStatsDTO: postLikeStats,
                            ),
                          if (postCommentStats.isDTO)
                            GraphItem(
                              title: 'Post comments',
                              color: 'rgb(54, 162, 235)',
                              tableStatsDTO: postCommentStats,
                            ),
                          if (hashtagStats.isDTO)
                            GraphItem(
                              title: 'Hashtags',
                              color: 'rgb(201, 203, 207)',
                              tableStatsDTO: hashtagStats,
                            ),
                        ],
                      ),
                    if (userStats.isDTO)
                      _buildGraph(
                        title: 'Users',
                        type: 'line',
                        graphItems: [
                          GraphItem(title: 'Users', color: 'rgb(75, 192, 192)', tableStatsDTO: userStats),
                        ],
                      ),
                    if (postStats.isDTO)
                      _buildGraph(
                        title: 'Posts',
                        type: 'line',
                        graphItems: [
                          GraphItem(title: 'Posts', color: 'rgb(255, 159, 64)', tableStatsDTO: postStats),
                        ],
                      ),
                  ],
                ),

                // right side containing stats

                Division(
                  className: 'col-md-12 col-lg-4',
                  child: Division(
                    className: 'card border-0 p-0 ml-xl-0 ml-lg-0',
                    children: [
                      const Division(
                        className: 'px-card py-10',
                        child: Heading2(className: 'card-title font-size-18 m-0', innerText: 'Stats'),
                      ),

                      // seprater

                      const HorizontalRule(),

                      // stats

                      Division(
                        className: 'py-15',
                        children: [
                          if (userStats.isDTO) _buildTotalStatsItem(userStats, 'Users'),
                          if (postStats.isDTO) _buildTotalStatsItem(postStats, 'Posts'),
                          if (hashtagStats.isDTO) _buildTotalStatsItem(hashtagStats, 'Hashtags'),
                          if (postLikeStats.isDTO) _buildTotalStatsItem(postLikeStats, 'Post likes'),
                          if (postCommentStats.isDTO) _buildTotalStatsItem(postCommentStats, 'Post comments'),
                        ],
                      ),
                    ],
                  ),
                ),
              ],
            ),
          );
        }

        return UtilWidgets.preloader('Crunching latest data...');
      },
    );
  }

  Widget _buildChartTitle(String title) {
    return Division(
      className: 'px-car py-10',
      child: Heading2(
        className: 'card-title font-size-18 m-0',
        innerText: title,
      ),
    );
  }

  Widget _buildGraph({
    required String title,
    required String type,
    required List<GraphItem> graphItems,
  }) {
    if (graphItems.isEmpty) return const Text('');

    return Division(
      className: 'card border-0 py-0',
      children: [
        _buildChartTitle(title),
        const HorizontalRule(),
        Division(
          className: 'p-10',
          child: Canvas(id: 'canvas-${title.toLowerCase()}-stats-chart', className: 'w-full'),
        ),
        RawMarkUp('''<script>
                new Chart(
                    document.getElementById('canvas-${title.toLowerCase()}-stats-chart'),
                    {
                      type: '$type',
                      data: {
                          labels: ["${graphItems.first.tableStatsDTO.previousSixMonthStats.keys.join('", "')}"],
                          datasets: [
                            ${graphItems.map((e) => '''
                                  {
                                    label: '${e.title}',
                                    backgroundColor: '${e.color}',
                                    borderColor: '${e.color}',
                                    data: [${e.tableStatsDTO.previousSixMonthStats.values.join(', ')}],
                                    tension: 0.2,
                                  }
                                ''').join(', ')}
                          ]
                      },
                      options: {
                          plugins: {
                              legend: {
                                  display: ${graphItems.length > 1 ? 'true' : 'false'}
                              }
                          }
                      }
                    }
                );
            </script>'''),
      ],
    );
  }

  Widget _buildTotalStatsItem(AdminTableStatsDTO dto, String title) {
    return Division(
      id: 'total_stats_item_$title',
      className: 'px-card py-10',
      children: [
        Span(
          className: 'card-title font-size-24',
          children: [
            Text(dto.totalRows),
            Span(className: 'text-muted font-size-14 mx-5', innerText: title),
          ],
        ),
        Division(
          className: 'font-size-12 text-muted my-10',
          children: [
            Span(className: 'font-italic text-success', innerText: '${dto.percentageFromLastMonth}%'),
            const Text(' from last month'),
          ],
        ),
        const HorizontalRule(),
      ],
    );
  }
}
